'use client';

import { useQuizStore } from '@/hooks/use-quiz-store';
import { Card, CardContent, CardHeader, CardTitle, CardFooter, CardDescription } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { PieChart, Pie, Cell, Tooltip, ResponsiveContainer, Legend } from 'recharts';
import { BrainCircuit, Check, X, Percent, BookCopy } from 'lucide-react';
import { Alert, AlertDescription, AlertTitle } from '@/components/ui/alert';
import Link from 'next/link';

const COLORS = ['hsl(var(--chart-1))', 'hsl(var(--chart-3))'];

const StatCard = ({ title, value, icon }: { title: string, value: string | number, icon: React.ReactNode }) => (
    <Card className="shadow-sm">
        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium text-muted-foreground">{title}</CardTitle>
            {icon}
        </CardHeader>
        <CardContent>
            <div className="text-2xl font-bold">{value}</div>
        </CardContent>
    </Card>
);

export default function PerformanceClient() {
    const { stats, isInitialized } = useQuizStore();

    if (!isInitialized) {
        return <div className="flex justify-center items-center h-64"><BrainCircuit className="h-12 w-12 animate-pulse text-primary" /></div>;
    }
    
    const answeredCount = stats.correct + stats.incorrect;
    const accuracy = answeredCount > 0 ? ((stats.correct / answeredCount) * 100).toFixed(1) : '0.0';
    
    const chartData = [
        { name: 'Correct', value: stats.correct },
        { name: 'Incorrect', value: stats.incorrect },
    ];
    
    const hasData = answeredCount > 0;

    return (
        <div className="space-y-8">
            {hasData ? (
                <>
                    <Card className="shadow-lg">
                        <CardHeader>
                            <CardTitle className="font-headline text-2xl">Quiz Attempt Results</CardTitle>
                            <CardDescription className="text-lg">Category: <span className="font-semibold text-primary">{stats.category}</span></CardDescription>
                        </CardHeader>
                    </Card>

                    <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
                        <StatCard title="Total Questions in Quiz" value={stats.total} icon={<BookCopy className="h-4 w-4 text-muted-foreground" />} />
                        <StatCard title="Answered Questions" value={answeredCount} icon={<BrainCircuit className="h-4 w-4 text-muted-foreground" />} />
                        <StatCard title="Correct Answers" value={stats.correct} icon={<Check className="h-4 w-4 text-green-500" />} />
                        <StatCard title="Incorrect Answers" value={stats.incorrect} icon={<X className="h-4 w-4 text-red-500" />} />
                    </div>

                    <Card className="shadow-lg">
                        <CardHeader>
                            <CardTitle className="font-headline">Score Breakdown</CardTitle>
                            <CardDescription>Accuracy: {accuracy}%</CardDescription>
                        </CardHeader>
                        <CardContent>
                            <div className="w-full h-[300px]">
                                <ResponsiveContainer>
                                    <PieChart>
                                        <Pie
                                            data={chartData}
                                            cx="50%"
                                            cy="50%"
                                            labelLine={false}
                                            outerRadius={120}
                                            fill="#8884d8"
                                            dataKey="value"
                                            label={({ cx, cy, midAngle, innerRadius, outerRadius, value, index }) => {
                                                if (value === 0) return null;
                                                const RADIAN = Math.PI / 180;
                                                const radius = innerRadius + (outerRadius - innerRadius) * 0.5;
                                                const x = cx + radius * Math.cos(-midAngle * RADIAN);
                                                const y = cy + radius * Math.sin(-midAngle * RADIAN);

                                                return (
                                                    <text x={x} y={y} fill="white" textAnchor={x > cx ? 'start' : 'end'} dominantBaseline="central">
                                                        {value}
                                                    </text>
                                                );
                                            }}
                                        >
                                            {chartData.map((entry, index) => (
                                                <Cell key={`cell-${index}`} fill={COLORS[index % COLORS.length]} />
                                            ))}
                                        </Pie>
                                        <Tooltip
                                            contentStyle={{
                                                backgroundColor: 'hsl(var(--background))',
                                                border: '1px solid hsl(var(--border))',
                                                borderRadius: 'var(--radius)'
                                            }}
                                        />
                                        <Legend />
                                    </PieChart>
                                </ResponsiveContainer>
                            </div>
                        </CardContent>
                    </Card>
                </>
            ) : (
                <Card>
                    <CardHeader>
                         <CardTitle>No Performance Data</CardTitle>
                         <CardDescription>You haven't completed a quiz attempt yet.</CardDescription>
                    </CardHeader>
                    <CardContent>
                        <Alert>
                            <BrainCircuit className="h-4 w-4" />
                            <AlertTitle>Let's Get Started!</AlertTitle>
                            <AlertDescription>
                                Head back to the main page to start a new quiz. Your results will show up here once you finish an attempt.
                            </AlertDescription>
                        </Alert>
                    </CardContent>
                    <CardFooter>
                       <Link href="/" className="w-full">
                         <Button className="w-full">Start a New Quiz</Button>
                       </Link>
                    </CardFooter>
                </Card>
            )}
        </div>
    );
}
