
'use client';

import { useState, useMemo, useEffect } from 'react';
import Link from 'next/link';
import Image from 'next/image';
import { useRouter } from 'next/navigation';
import { useQuizStore } from '@/hooks/use-quiz-store';
import { Card, CardContent, CardDescription, CardFooter, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { RadioGroup, RadioGroupItem } from '@/components/ui/radio-group';
import { Label } from '@/components/ui/label';
import { CheckCircle, XCircle, Trophy, BrainCircuit, FileQuestion, Lightbulb, Grid3x3, Flag, Clock } from 'lucide-react';
import { cn } from '@/lib/utils';
import { PlaceHolderImages } from '@/lib/placeholder-images';
import { Alert, AlertDescription, AlertTitle } from '../ui/alert';
import { Badge } from '../ui/badge';
import type { Question } from '@/lib/types';
import { Sheet, SheetContent, SheetHeader, SheetTitle, SheetTrigger } from '../ui/sheet';

interface QuizClientProps {
  questions: Question[];
  category: string;
  quizType: 'practice' | 'timed';
}

function QuizTimer({ onTimeUp }: { onTimeUp: () => void }) {
  const { timeLimit, startTime } = useQuizStore();
  const [remainingTime, setRemainingTime] = useState(0);

  useEffect(() => {
    if (startTime === null || timeLimit === null) return;
    
    const endTime = startTime + timeLimit * 60 * 1000;

    const interval = setInterval(() => {
      const now = Date.now();
      const timeLeft = Math.max(0, endTime - now);
      setRemainingTime(timeLeft);

      if (timeLeft === 0) {
        clearInterval(interval);
        onTimeUp();
      }
    }, 1000);

    return () => clearInterval(interval);
  }, [startTime, timeLimit, onTimeUp]);

  if (timeLimit === null) return null;

  const minutes = Math.floor(remainingTime / (1000 * 60));
  const seconds = Math.floor((remainingTime % (1000 * 60)) / 1000);

  return (
    <Card className="mb-4 shadow-lg">
      <CardHeader className="flex-row items-center justify-between p-4">
        <div className="flex items-center gap-2">
            <Clock className="h-6 w-6 text-primary" />
            <CardTitle className="font-headline text-xl">Time Remaining</CardTitle>
        </div>
        <div className="text-2xl font-bold font-mono bg-primary/10 text-primary px-4 py-2 rounded-lg">
            {String(minutes).padStart(2, '0')}:{String(seconds).padStart(2, '0')}
        </div>
      </CardHeader>
    </Card>
  );
}

function QuestionNavigator({ questions, userAnswers, currentQuestionIndex, goToQuestion, handleFinishAttempt, allQuestionsAnswered, flaggedQuestions }: any) {
    return (
        <Card className="shadow-lg lg:sticky lg:top-20">
            <CardHeader>
                <CardTitle className="font-headline text-xl">Questions</CardTitle>
            </CardHeader>
            <CardContent>
                <div className="grid grid-cols-5 gap-2">
                    {questions.map((q: Question, index: number) => {
                        const isAnswered = userAnswers.hasOwnProperty(q.id);
                        const isCurrent = index === currentQuestionIndex;
                        const isFlagged = flaggedQuestions.includes(q.id);
                        
                        let variant: "default" | "secondary" | "outline" | "ghost" | "link" = isCurrent ? "default" : "outline";
                        
                        return (
                            <Button
                                key={q.id}
                                variant={variant}
                                size="icon"
                                className={cn(
                                    "h-10 w-10 text-base font-bold relative",
                                    isCurrent ? "ring-2 ring-primary ring-offset-2" : "border",
                                    isAnswered && !isCurrent && "bg-muted text-muted-foreground",
                                    !isAnswered && !isCurrent && "bg-card hover:bg-accent"
                                )}
                                onClick={() => goToQuestion(index)}
                            >
                                {isFlagged && (
                                    <div className="absolute top-0 right-0 w-0 h-0 border-t-[10px] border-t-red-500 border-l-[10px] border-l-transparent"></div>
                                )}
                                {index + 1}
                            </Button>
                        );
                    })}
                </div>
                <Button 
                    onClick={handleFinishAttempt}
                    className={cn(
                        'w-full mt-6',
                        allQuestionsAnswered ? 'bg-green-600 hover:bg-green-700' : 'bg-destructive hover:bg-destructive/90'
                    )}
                >
                    Finish Attempt
                </Button>
            </CardContent>
        </Card>
    )
}


export default function QuizClient({ questions, category, quizType }: QuizClientProps) {
  const { recordAnswer, isInitialized, userAnswers, saveUserAnswer, resetUserAnswers, flaggedQuestions, toggleFlag } = useQuizStore();
  const [currentQuestionIndex, setCurrentQuestionIndex] = useState(0);
  const [selectedAnswer, setSelectedAnswer] = useState<string | null>(null);
  const [isSubmitted, setIsSubmitted] = useState(false);
  const [shake, setShake] = useState(false);
  const router = useRouter();

  const quizCompleteImage = useMemo(() => PlaceHolderImages.find(p => p.id === 'quiz-complete'), []);
  const storageKey = useMemo(() => `medquiz-progress-${category}`, [category]);
  const currentQuestion = useMemo(() => questions[currentQuestionIndex], [questions, currentQuestionIndex]);
  const isCurrentQuestionFlagged = useMemo(() => currentQuestion && flaggedQuestions.includes(currentQuestion.id), [currentQuestion, flaggedQuestions]);

  useEffect(() => {
    try {
      const savedProgress = localStorage.getItem(storageKey);
      if (savedProgress) {
        const { savedIndex } = JSON.parse(savedProgress);
        if (savedIndex < questions.length) {
            setCurrentQuestionIndex(savedIndex);
        }
      }
    } catch (error) {
        console.error("Failed to load quiz progress from localStorage", error);
    }
  }, [questions, storageKey]);


  useEffect(() => {
    if (isInitialized) {
        try {
            const progress = {
                savedIndex: currentQuestionIndex,
            };
            localStorage.setItem(storageKey, JSON.stringify(progress));
        } catch (error) {
            console.error("Failed to save quiz progress to localStorage", error);
        }
    }
  }, [currentQuestionIndex, storageKey, isInitialized]);


  useEffect(() => {
    if (currentQuestion && userAnswers[currentQuestion.id]) {
      setSelectedAnswer(userAnswers[currentQuestion.id]);
      setIsSubmitted(true);
    } else {
      setSelectedAnswer(null);
      setIsSubmitted(false);
    }
  }, [currentQuestion, userAnswers]);


  const allQuestionsAnswered = useMemo(() => {
    return Object.keys(userAnswers).length === questions.length;
  }, [userAnswers, questions.length]);

  const handleFinishAttempt = () => {
    router.push('/performance');
  }

  if (!isInitialized) {
    return <div className="flex justify-center items-center h-64"><BrainCircuit className="h-12 w-12 animate-pulse text-primary" /></div>;
  }
  
  if (questions.length === 0) {
    return (
        <Card className="text-center shadow-lg">
            <CardHeader className="items-center">
                 <div className="bg-secondary rounded-lg p-3 w-fit h-fit">
                    <FileQuestion className="h-12 w-12 text-secondary-foreground" />
                </div>
                <CardTitle className="font-headline text-2xl pt-4">No Questions Available</CardTitle>
                 <CardDescription>for &quot;{category}&quot;</CardDescription>
            </CardHeader>
            <CardContent>
                <p className="text-muted-foreground">There are no questions in this category. <br/>Please add some questions to get started.</p>
            </CardContent>
            <CardFooter>
                <Link href="/manage" className="w-full">
                    <Button className="w-full">Manage Questions</Button>
                </Link>
            </CardFooter>
        </Card>
    );
  }

  const handleSubmit = () => {
    if (!selectedAnswer) return;
    
    const isAlreadyAnswered = userAnswers.hasOwnProperty(currentQuestion.id);
  
    if (!isAlreadyAnswered) {
        const correct = selectedAnswer === currentQuestion.correctAnswer;
        recordAnswer(correct);
        if (!correct) {
          setShake(true);
          setTimeout(() => setShake(false), 500);
        }
    }

    saveUserAnswer(currentQuestion.id, selectedAnswer);
    setIsSubmitted(true);
  };

  const handleNext = () => {
    setIsSubmitted(false);
    setSelectedAnswer(null);
    setCurrentQuestionIndex(prev => prev + 1);
  };
  
  const handleRestart = () => {
    try {
        localStorage.removeItem(storageKey);
    } catch(error) {
        console.error("Failed to remove quiz progress from localStorage", error);
    }
    setCurrentQuestionIndex(0);
    setIsSubmitted(false);
    setSelectedAnswer(null);
    resetUserAnswers();
  }

  const goToQuestion = (index: number) => {
    if (index >= 0 && index < questions.length) {
      setCurrentQuestionIndex(index);
    }
  }

  const handleFlag = () => {
    if(currentQuestion) {
        toggleFlag(currentQuestion.id);
    }
  }

  if (currentQuestionIndex >= questions.length) {
    return (
      <Card className="text-center shadow-lg overflow-hidden">
        {quizCompleteImage && (
          <div className="relative h-48 w-full">
            <Image 
              src={quizCompleteImage.imageUrl}
              alt={quizCompleteImage.description}
              data-ai-hint={quizCompleteImage.imageHint}
              fill
              className="object-cover"
              priority
            />
          </div>
        )}
        <CardHeader className="pt-6">
          <div className="mx-auto bg-card/80 backdrop-blur-sm rounded-full p-4 w-fit -mt-20 relative z-10 border shadow-lg">
            <Trophy className="h-12 w-12 text-primary" />
          </div>
          <CardTitle className="font-headline text-3xl mt-4">Quiz Complete!</CardTitle>
          <CardDescription>You finished the &quot;{category}&quot; quiz.</CardDescription>
        </CardHeader>
        <CardContent>
          <p className="text-lg">You have completed all the questions.</p>
          <p className="text-muted-foreground mt-2">Check your performance to see how you did.</p>
        </CardContent>
        <CardFooter className="flex-col sm:flex-row gap-2">
            <Button onClick={handleRestart} variant="outline" className="w-full">Restart Quiz</Button>
            <Link href="/performance" className="w-full">
                <Button className="w-full">View Performance</Button>
            </Link>
        </CardFooter>
      </Card>
    );
  }

  return (
    <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
      <div className="lg:col-span-2">
        <Card className={cn("w-full shadow-lg transition-transform duration-500", shake && "animation-shake")}>
          <CardHeader>
            <div className="flex justify-between items-center mb-2">
              <CardDescription>Question {currentQuestionIndex + 1} of {questions.length}</CardDescription>
              <Badge variant="secondary">{currentQuestion.category}</Badge>
            </div>
            <CardTitle className="font-headline text-2xl leading-relaxed whitespace-pre-wrap">{currentQuestion.questionText}</CardTitle>
          </CardHeader>
          <CardContent>
            <RadioGroup value={selectedAnswer ?? ''} onValueChange={setSelectedAnswer} disabled={isSubmitted}>
              <div className="space-y-4">
                {currentQuestion.options.map((option) => {
                  const isSelected = selectedAnswer === option;
                  const isCorrectOption = option === currentQuestion.correctAnswer;
                  
                  return (
                    <Label
                      key={option}
                      className={cn(
                        "flex items-center rounded-lg border p-4 text-base transition-all cursor-pointer hover:bg-secondary/50",
                        isSubmitted && isCorrectOption && "border-green-500 bg-green-500/10 text-green-800 dark:text-green-400 dark:border-green-400 dark:bg-green-500/10",
                        isSubmitted && isSelected && !isCorrectOption && "border-red-500 bg-red-500/10 text-red-800 dark:text-red-400 dark:border-red-400 dark:bg-red-500/10",
                        isSubmitted && !isCorrectOption && !isSelected && "text-muted-foreground opacity-70"
                      )}
                    >
                      <RadioGroupItem value={option} id={option} className="mr-4" />
                      <span className="whitespace-pre-wrap">{option}</span>
                       {isSubmitted && isCorrectOption && <CheckCircle className="ml-auto h-5 w-5 text-green-500" />}
                       {isSubmitted && isSelected && !isCorrectOption && <XCircle className="ml-auto h-5 w-5 text-red-500" />}
                    </Label>
                  );
                })}
              </div>
            </RadioGroup>
            {isSubmitted && (
              <Alert className="mt-6">
                <Lightbulb className="h-4 w-4" />
                <AlertTitle className="font-headline">Explanation</AlertTitle>
                <AlertDescription className="whitespace-pre-wrap">
                  {currentQuestion.explanation}
                </AlertDescription>
              </Alert>
            )}
          </CardContent>
          <CardFooter className="flex-col sm:flex-row gap-2">
            {isSubmitted ? (
                <Button onClick={handleNext} className="w-full">
                    {currentQuestionIndex === questions.length - 1 ? "Finish Quiz" : "Next Question"}
                </Button>
            ) : (
                <Button onClick={handleSubmit} disabled={!selectedAnswer} className="w-full">
                    Submit Answer
                </Button>
            )}
             <Button variant={isCurrentQuestionFlagged ? "destructive" : "outline"} onClick={handleFlag} className="sm:w-auto">
                <Flag className="mr-2 h-4 w-4" />
                {isCurrentQuestionFlagged ? "Unflag" : "Flag"}
            </Button>
          </CardFooter>
        </Card>
        <div className="lg:hidden mt-4">
             <Sheet>
                <SheetTrigger asChild>
                    <Button variant="outline" className="w-full">
                        <Grid3x3 className="mr-2 h-4 w-4" />
                        Show Questions
                    </Button>
                </SheetTrigger>
                <SheetContent side="bottom" className="rounded-t-lg">
                    <SheetHeader className="mb-4">
                        <SheetTitle className="font-headline text-xl">Questions</SheetTitle>
                    </SheetHeader>
                    <div className="grid grid-cols-5 gap-2">
                        {questions.map((q, index) => {
                            const isAnswered = userAnswers.hasOwnProperty(q.id);
                            const isCurrent = index === currentQuestionIndex;
                            const isFlagged = flaggedQuestions.includes(q.id);

                            let variant: "default" | "secondary" | "outline" | "ghost" | "link" = isCurrent ? "default" : "outline";

                            return (
                                <Button
                                    key={q.id}
                                    variant={variant}
                                    size="icon"
                                    className={cn(
                                        "h-10 w-10 text-base font-bold relative",
                                        isCurrent ? "ring-2 ring-primary ring-offset-2" : "border",
                                        isAnswered && !isCurrent && "bg-muted text-muted-foreground",
                                        !isAnswered && !isCurrent && "bg-card hover:bg-accent"
                                    )}
                                    onClick={() => goToQuestion(index)}
                                >
                                     {isFlagged && (
                                        <div className="absolute top-0 right-0 w-0 h-0 border-t-[10px] border-t-red-500 border-l-[10px] border-l-transparent"></div>
                                    )}
                                    {index + 1}
                                </Button>
                            );
                        })}
                    </div>
                </SheetContent>
            </Sheet>
        </div>
      </div>
      <div className="hidden lg:block">
        {quizType === 'timed' && <QuizTimer onTimeUp={handleFinishAttempt} />}
        <QuestionNavigator 
            questions={questions}
            userAnswers={userAnswers}
            currentQuestionIndex={currentQuestionIndex}
            goToQuestion={goToQuestion}
            handleFinishAttempt={handleFinishAttempt}
            allQuestionsAnswered={allQuestionsAnswered}
            flaggedQuestions={flaggedQuestions}
        />
      </div>
    </div>
  );
}
