
'use client';

import { create } from 'zustand';
import { createJSONStorage, persist } from 'zustand/middleware';
import type { Question, PerformanceStats } from '@/lib/types';

import adolescentMedicine from '@/lib/questions/adolescent-medicine.json';
import allergicDisorders from '@/lib/questions/allergic-disorders.json';
import behavioralAndPsychiatricDisorders from '@/lib/questions/behavioral-and-psychiatric-disorders.json';
import boneAndJointDisorders from '@/lib/questions/bone-and-joint-disorders.json';
import cancerAndBenignTumors from '@/lib/questions/cancer-and-benign-tumors.json';
import diseasesOfTheBlood from '@/lib/questions/diseases-of-the-blood.json';
import disordersOfTheEye from '@/lib/questions/disorders-of-the-eye.json';
import emergencyMedicineAndCriticalCare from '@/lib/questions/emergency-medicine-and-critical-care.json';
import environmentalHealth from '@/lib/questions/environmental-health.json';
import fluidAndElectrolyteDisorders from '@/lib/questions/fluid-and-electrolyte-disorders.json';
import growthDevelopmentAndBehavior from '@/lib/questions/growth-development-and-behavior.json';
import gynecologicProblemsOfChildhood from '@/lib/questions/gynecologic-problems-of-childhood.json';
import humanGenetics from '@/lib/questions/human-genetics.json';
import immunology from '@/lib/questions/immunology.json';
import infectiousDiseases from '@/lib/questions/infectious-diseases.json';
import laboratoryMedicine from '@/lib/questions/laboratory-medicine.json';
import learningAndDevelopmentalDisorders from '@/lib/questions/learning-and-developmental-disorders.json';
import metabolicDisorders from '@/lib/questions/metabolic-disorders.json';
import nephrology from '@/lib/questions/nephrology.json';
import neuromuscularDisorders from '@/lib/questions/neuromuscular-disorders.json';
import nutrition from '@/lib/questions/nutrition.json';
import pediatricDrugTherapy from '@/lib/questions/pediatric-drug-therapy.json';
import rehabilitationMedicine from '@/lib/questions/rehabilitation-medicine.json';
import rheumaticDiseasesOfChildhood from '@/lib/questions/rheumatic-diseases-of-childhood.json';
import skin from '@/lib/questions/skin.json';
import theCardiovascularSystem from '@/lib/questions/the-cardiovascular-system.json';
import theDigestiveSystem from '@/lib/questions/the-digestive-system.json';
import theEar from '@/lib/questions/the-ear.json';
import theEndocrineSystem from '@/lib/questions/the-endocrine-system.json';
import theFetusAndTheNeonatalInfant from '@/lib/questions/the-fetus-and-the-neonatal-infant.json';
import theFieldOfPediatrics from '@/lib/questions/the-field-of-pediatrics.json';
import theNervousSystem from '@/lib/questions/the-nervous-system.json';
import theRespiratorySystem from '@/lib/questions/the-respiratory-system.json';
import urologicDisorders from '@/lib/questions/urologic-disorders.json';

const allQuestionData: Question[] = [
  ...adolescentMedicine,
  ...allergicDisorders,
  ...behavioralAndPsychiatricDisorders,
  ...boneAndJointDisorders,
  ...cancerAndBenignTumors,
  ...diseasesOfTheBlood,
  ...disordersOfTheEye,
  ...emergencyMedicineAndCriticalCare,
  ...environmentalHealth,
  ...fluidAndElectrolyteDisorders,
  ...growthDevelopmentAndBehavior,
  ...gynecologicProblemsOfChildhood,
  ...humanGenetics,
  ...immunology,
  ...infectiousDiseases,
  ...laboratoryMedicine,
  ...learningAndDevelopmentalDisorders,
  ...metabolicDisorders,
  ...nephrology,
  ...neuromuscularDisorders,
  ...nutrition,
  ...pediatricDrugTherapy,
  ...rehabilitationMedicine,
  ...rheumaticDiseasesOfChildhood,
  ...skin,
  ...theCardiovascularSystem,
  ...theDigestiveSystem,
  ...theEar,
  ...theEndocrineSystem,
  ...theFetusAndTheNeonatalInfant,
  ...theFieldOfPediatrics,
  ...theNervousSystem,
  ...theRespiratorySystem,
  ...urologicDisorders
];

const allQuestions = allQuestionData.map((q, index) => ({ ...q, id: `${q.category}-${index}`, category: q.category || 'General' }));

interface QuizState {
  questions: Question[];
  activeQuestions: Question[];
  stats: PerformanceStats;
  userAnswers: Record<string, string | null>;
  flaggedQuestions: string[];
  quizType: 'practice' | 'timed';
  timeLimit: number | null;
  startTime: number | null;
  addQuestion: (question: Omit<Question, 'id'>) => void;
  updateQuestion: (updatedQuestion: Question) => void;
  removeQuestion: (id: string) => void;
  startNewQuizAttempt: (category: string) => void;
  startRandomQuiz: (count: number, categories: string[], quizType: 'practice' | 'timed', timeLimit: number | null) => void;
  recordAnswer: (isCorrect: boolean) => void;
  saveUserAnswer: (questionId: string, answer: string) => void;
  toggleFlag: (questionId: string) => void;
  resetStats: () => void;
  resetUserAnswers: () => void;
  isInitialized: boolean;
  setInitialized: (isInitialized: boolean) => void;
}

const useQuizStore = create<QuizState>()(
  persist(
    (set, get) => ({
      questions: allQuestions,
      activeQuestions: [],
      stats: { correct: 0, incorrect: 0, total: 0, category: 'N/A' },
      userAnswers: {},
      flaggedQuestions: [],
      quizType: 'practice',
      timeLimit: null,
      startTime: null,
      isInitialized: false,
      setInitialized: (isInitialized: boolean) => set({ isInitialized }),
      
      startNewQuizAttempt: (category) => {
        const questionsForCategory = category === 'All Questions' 
          ? get().questions 
          : get().questions.filter(q => q.category === category);
        
        set({ 
          activeQuestions: questionsForCategory,
          stats: { correct: 0, incorrect: 0, total: questionsForCategory.length, category },
          userAnswers: {},
          flaggedQuestions: [],
          quizType: 'practice',
          timeLimit: null,
          startTime: null,
        });
      },

      startRandomQuiz: (count, categories, quizType, timeLimit) => {
        const filtered = categories.length > 0 
          ? allQuestions.filter(q => categories.includes(q.category))
          : allQuestions;
        
        // Shuffle the array
        const shuffled = filtered.sort(() => 0.5 - Math.random());
        
        // Get sub-array of first n elements after shuffling
        const selected = shuffled.slice(0, Math.min(count, shuffled.length));

        set({
            activeQuestions: selected,
            stats: { correct: 0, incorrect: 0, total: selected.length, category: 'Random Quiz' },
            userAnswers: {},
            flaggedQuestions: [],
            quizType: quizType,
            timeLimit: quizType === 'timed' ? timeLimit : null,
            startTime: quizType === 'timed' ? Date.now() : null,
        });
      },

      addQuestion: (question) => {
        const newQuestion = { ...question, id: `q${Date.now()}` };
        set((state) => ({ questions: [...state.questions, newQuestion] }));
      },
      updateQuestion: (updatedQuestion) => {
        set((state) => ({
          questions: state.questions.map((q) => (q.id === updatedQuestion.id ? updatedQuestion : q)),
        }));
      },
      removeQuestion: (id) => {
        set((state) => ({ questions: state.questions.filter((q) => q.id !== id) }));
      },
      recordAnswer: (isCorrect: boolean) => {
        set((state) => ({
          stats: {
            ...state.stats,
            correct: isCorrect ? state.stats.correct + 1 : state.stats.correct,
            incorrect: !isCorrect ? state.stats.incorrect + 1 : state.stats.incorrect,
          },
        }));
      },
      saveUserAnswer: (questionId, answer) => {
        set((state) => ({
          userAnswers: {
            ...state.userAnswers,
            [questionId]: answer,
          },
        }));
      },
      toggleFlag: (questionId: string) => {
        set((state) => {
            const isFlagged = state.flaggedQuestions.includes(questionId);
            if (isFlagged) {
                return { flaggedQuestions: state.flaggedQuestions.filter(id => id !== questionId) };
            } else {
                return { flaggedQuestions: [...state.flaggedQuestions, questionId] };
            }
        });
      },
      resetStats: () => {
        set({ 
          stats: { correct: 0, incorrect: 0, total: 0, category: 'N/A' },
          quizType: 'practice',
          timeLimit: null,
          startTime: null,
         });
      },
      resetUserAnswers: () => {
        set({ userAnswers: {}, flaggedQuestions: [] });
      },
    }),
    {
      name: 'medquiz-storage',
      storage: createJSONStorage(() => localStorage),
      partialize: (state) => ({ 
          stats: state.stats, 
          userAnswers: state.userAnswers,
          flaggedQuestions: state.flaggedQuestions,
          quizType: state.quizType,
          timeLimit: state.timeLimit,
          startTime: state.startTime,
      }),
      onRehydrateStorage: () => (state) => {
        if (state) {
          state.setInitialized(true);
        }
      },
    }
  )
);

// This is a bit of a workaround to ensure the store is initialized on the client
// before any components try to use it.
if (typeof window !== 'undefined') {
  // Merge allQuestionData into the state on initialization, but only if not already there from persistence
  const currentState = useQuizStore.getState();
  if (currentState.questions.length === 0) {
      useQuizStore.setState({ questions: allQuestions, isInitialized: true });
  } else {
      // If questions are already there (from persist), just set initialized
      useQuizStore.setState({ isInitialized: true });
  }
}

export { useQuizStore };
